package easik.sketch.util.graph;

import java.util.Vector;

import org.jgraph.graph.DefaultCellViewFactory;
import org.jgraph.graph.EdgeView;
import org.jgraph.graph.VertexView;

/**
 * A class that can be extended in order to add new ways of rendering cells.
 * A cell is the most basic form of data component in JGraph, it can be either an edge or a vertex.
 * In order to create a new drawing form, a new view must be created, along with a renderer to be associated with it.
 * The renderer must override the <code>getPreferredSize</code> and <code>paint</code> methods.
 * It is recommended that you map new cell types to views, instead of mapping new views to existing cell types.
 * It is unspecified what will happen if two views are mapped to the same cell type.
 * Once the cell and view classes are created, just call <code>addCellType</code> to create a mapping between the cell and view classes.
 * 
 * @author William Belanger 2006
 * @since 2006-07-25 William Belanger
 * @version 2006-07-25 William Belanger
 */
public abstract class ExtendableCellViewFactory extends DefaultCellViewFactory {
	
	/** The list of mappings between cell types and view types */
	private Vector<CellViewMapping> viewIndirections;
	
	/**
	 * Default constructor that adds no new cell to view mappings.
	 */
	public ExtendableCellViewFactory(){
		viewIndirections = new Vector<CellViewMapping>(15);
	}
	
	/**
	 * Adds a cell to view mapping to the list of mappings.
	 * @param cell The cell type.
	 * @param view The view type that should be used when creating a drawing of this cell type.
	 */
	public void addCellType(Class cell, Class view){
		CellViewMapping indir = new CellViewMapping();
		indir.setCellClass(cell);
		indir.setViewClass(view);
		viewIndirections.add(indir);
	}
	
	/**
	 * A mapping of cell classes to view classes.
	 * An instance of this class is added through the <code>addCellType(Class, Class)</code> method.
	 * When a cell is to be rendered, the view type associated with it through an object of this class is used.
	 * @author code taken from an example that comes with JGraph
	 * @since 2006-05-23 William Belanger
	 * @version 2006-05-23 William Belanger
	 */
	private static class CellViewMapping{
		
		/**
		 * The cell class
		 */
		private Class cellClass;
		/**
		 * The view class
		 */
		private Class viewClass;
		
		/**
		 * @return The class of the cell.
		 */
		public Class getCellClass(){
			return cellClass;
		}
		/**
		 * Sets the class of the cell.
		 * @param c The class to set the cell as having.
		 */
		public void setCellClass(Class c){
			cellClass = c;
		}
		/**
		 * @return The class of the view.
		 */
		public Class getViewClass(){
			return viewClass;
		}
		/**
		 * Sets the class of the view.
		 * @param v The class to set the view as having.
		 */
		public void setViewClass(Class v){
			viewClass = v;
		}
	}
	
	/**
	 * Reads through the list of registered mappings to see if this object is a cell of the specified type.
	 * 
	 * @param v The object to base creation on
	 * @return The view for the object to be rendered in
	 */
	protected VertexView createVertexView(Object v){
		VertexView view = null;
		try{
			for(CellViewMapping indr : getViewIndirections()){
				if(v.getClass() == indr.getCellClass()){
					view = (VertexView)indr.getViewClass().newInstance();
					view.setCell(v);
					return view;
				}
			}
		} catch(IllegalAccessException e){
			e.printStackTrace();
		} catch(InstantiationException e){
			e.printStackTrace();
		}
		return super.createVertexView(v);
	}
	
	/**
	 * @return The list of mappings between cells and views.
	 */
	public Vector<CellViewMapping> getViewIndirections(){
		return viewIndirections;
	}
	/**
	 * @param list The new list of mappings between cells and views.
	 */
	public void setViewIndirections(Vector<CellViewMapping> list){
		viewIndirections = list;
	}
	
	/**
	 * Creates edge views
	 * 
	 * @param cell The edge cell to base creation on
	 * @return The created edge view
	 */
	protected EdgeView createEdgeView(Object cell) {
		EdgeView view = null;
		try{
			for(CellViewMapping indr : getViewIndirections()){
				if(cell.getClass() == indr.getCellClass()){
					view = (EdgeView)indr.getViewClass().newInstance();
					view.setCell(cell);
					return view;
				}
			}
		} catch(IllegalAccessException e){
			e.printStackTrace();
		} catch(InstantiationException e){
			e.printStackTrace();
		}
		return super.createEdgeView(cell);
    }
}

